/*
 * CRC64_calc.c
 *
 * Function to calculate CRC64 in software.
 *
 * Copyright (C) 2016 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "sys_common.h"
#include "CRC64_calc.h"

static __inline uint64 crc64_update_word(uint64 crc64, uint64 data)
{
    int    i, j;
    uint64 nextCrc = 0;

#if defined(__big_endian__)
    /* swap64 */
    if(1)
	{
    	uint32 tmp = 0UL;

		union {
			uint64 ll;
			uint32 l[2];
		} U64_;

		U64_.ll   = data;

		tmp       = U64_.l[0];
		U64_.l[0] = U64_.l[1];
		U64_.l[1] = tmp;

		data     = U64_.ll;
	}
#elif defined(__little_endian__)
	data = data;
#else
#error Endianess isn't defined'
#endif

    // for i in 63 to 0 loop
	#pragma MUST_ITERATE(64, 64);
	#pragma UNROLL(64);
    for(i = 63; i >= 0; i--)
    {
        // NEXT_CRC_VAL(0) := CRC_VAL(63) xor DATA(i);
        nextCrc = (nextCrc & 0xfffffffffffffffeULL) | ((crc64 >> 63) ^ (data >> i));

        // for j in 1 to 63 loop
		#pragma MUST_ITERATE(63, 63);
		#pragma UNROLL(63);
        for(j = 1; j < 64; j++)
        {
            // case j is
            // when 1|3|4 =>
            if(j == 1 || j == 3 || j == 4)
            {
                // NEXT_CRC_VAL(j) := CRC_VAL(j - 1) xor CRC_VAL(63) xor DATA(i);
                nextCrc = (nextCrc & ~(1ULL << j)) | ((((crc64 >> (j - 1)) ^ (crc64 >> 63) ^ (data >> i)) & 1) << j);
            }
            else
            { // when others =>
                // NEXT_CRC_VAL(j) := CRC_VAL(j - 1);
                nextCrc = (nextCrc & ~(1ULL << j)) | (((crc64 >> (j - 1)) & 1) << j);
            }
            // end case;
        } // end loop;
        crc64 = nextCrc;
    } // end loop

    return crc64;
}

uint64 calc_crc64(uint64* data, uint32 count)
{
	uint32 i     = 0UL;
	uint64 crc64 = 0ULL;


	for (i = 0UL ; i < count ; i++)
	{
		crc64 = crc64_update_word(crc64, data[i]);
	}

	/* swap64 */
	{
		uint32 tmp = 0UL;

		union {
			uint64 ll;
			uint32 l[2];
		} U64_;

		U64_.ll   = crc64;

		tmp       = U64_.l[0];
		U64_.l[0] = U64_.l[1];
		U64_.l[1] = tmp;

		crc64     = U64_.ll;
	}

	return crc64;
}

/* EOF */
